import shutil

import json

import os

import tkinter as tk

from tkinter import filedialog, messagebox, simpledialog

import subprocess

import sys

import webbrowser



# -------------------------

# Konstanter

# -------------------------



ADF_SIZE = 901120  # Standardstorlek för ADF-filer

APP_NAME = "ADF Commander"

APP_VERSION = "2.0"



# -------------------------

# ADF-skapande funktioner

# -------------------------



def create_blank_adf():

    """Skapa en tom oformaterad ADF-fil"""

    try:

        filename = filedialog.asksaveasfilename(

            defaultextension=".adf",

            filetypes=[("Amiga Disk Files", "*.adf")],

            title="Spara tom ADF-fil"

        )

        

        if not filename:

            return

        

        # Kontrollera om filen redan finns

        if os.path.exists(filename):

            if not messagebox.askyesno("Bekräfta", f"Filen {filename} finns redan. Vill du skriva över den?"):

                return

        

        # Skapa en tom fil med rätt storlek

        with open(filename, 'wb') as f:

            f.write(b'\x00' * ADF_SIZE)

        

        messagebox.showinfo("Klart!", 

            f"Tom ADF-fil skapad: {filename}\n\n"

            "OBS: Denna fil är inte formaterad med Amiga-filsystem. "

            "Den kommer fungera med de flesta emulatorer och kan "

            "formateras i Workbench om needed.")

        

        # Lägg till i listan

        adf_listbox.insert(tk.END, filename)

        update_status()

        

    except Exception as e:

        messagebox.showerror("Fel", f"Kunde inte skapa ADF-fil: {e}")



def create_formatted_adf():

    """Skapar en formaterad ADF-fil (OFS/FFS)"""

    filename = filedialog.asksaveasfilename(

        defaultextension=".adf",

        filetypes=[("Amiga Disk Files", "*.adf")],

        title="Spara formaterad ADF-fil"

    )

    if not filename:

        return

    

    try:

        # Först skapa en tom ADF

        with open(filename, 'wb') as f:

            f.write(b'\x00' * ADF_SIZE)

        

        # Försök formatera med xdftool om det finns

        try:

            subprocess.run(["xdftool", filename, "format", "Workbench", "ffs"], check=True)

            messagebox.showinfo("Klart!", f"Formaterad ADF skapad: {filename}")

        except (FileNotFoundError, subprocess.CalledProcessError):

            # Fallback om xdftool inte finns

            messagebox.showinfo(

                "Formaterade ADF-filer",

                "För att skapa formaterade ADF-filer behöver du:\n\n"

                "1. Installera xdftool eller amitools\n"

                "2. Eller använda externa verktyg som:\n"

                "   - ADFOpus\n"

                "   - WinUAE\n"

                "   - Direkt i Workbench\n\n"

                "En tom ADF har skapats som kan formateras senare."

            )

        

        adf_listbox.insert(tk.END, filename)

        update_status()

    except Exception as e:

        messagebox.showerror("Fel", f"Kunde inte skapa ADF: {e}")



# -------------------------

# ADF-redigeringsfunktioner

# -------------------------



def open_adf_editor():

    """Öppnar ett fönster för att redigera en ADF-fil"""

    filename = filedialog.askopenfilename(filetypes=[("Amiga Disk Files", "*.adf")])

    if not filename:

        return

    

    editor = tk.Toplevel(root)

    editor.title(f"Redigerar: {os.path.basename(filename)}")

    editor.geometry("700x500")



    # Main frame

    main_frame = tk.Frame(editor)

    main_frame.pack(fill=tk.BOTH, expand=True, padx=10, pady=10)



    # Listbox med scrollbar

    list_frame = tk.Frame(main_frame)

    list_frame.pack(side=tk.TOP, fill=tk.BOTH, expand=True)



    scrollbar = tk.Scrollbar(list_frame)

    scrollbar.pack(side=tk.RIGHT, fill=tk.Y)



    listbox = tk.Listbox(list_frame, width=70, height=20, yscrollcommand=scrollbar.set)

    listbox.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)

    scrollbar.config(command=listbox.yview)



    # Knappram

    btn_frame = tk.Frame(main_frame)

    btn_frame.pack(side=tk.BOTTOM, fill=tk.X, pady=5)



    def refresh_contents():

        listbox.delete(0, tk.END)

        try:

            # Kontrollera först om filen är en giltig ADF

            if os.path.getsize(filename) != ADF_SIZE:

                listbox.insert(tk.END, "[Ogiltig ADF-filstorlek]")

                return

                

            # Försök läsa med xdftool om det finns

            try:

                result = subprocess.run(

                    ["xdftool", filename, "list"],

                    capture_output=True, text=True, check=False

                )



                if result.returncode != 0:

                    listbox.insert(tk.END, "[ADF är inte formaterad eller ogiltig]")

                    return



                for line in result.stdout.splitlines():

                    listbox.insert(tk.END, line)

            except FileNotFoundError:

                listbox.insert(tk.END, "[Installera xdftool för att visa innehåll]")

                listbox.insert(tk.END, "[ADF-filen är skapad men kan inte inspekteras]")



        except Exception as e:

            messagebox.showerror("Fel", f"Kunde inte läsa ADF: {e}")



    def add_file():

        pc_file = filedialog.askopenfilename()

        if not pc_file:

            return

        amiga_name = simpledialog.askstring("Amiga-filnamn", "Filnamn i ADF:")

        if not amiga_name:

            return

        try:

            subprocess.run(["xdftool", filename, "write", pc_file, amiga_name], check=True)

            refresh_contents()

        except Exception as e:

            messagebox.showerror("Fel", f"Kunde inte lägga till fil: {e}")



    def extract_file():

        selection = listbox.curselection()

        if not selection:

            return

        

        selected_text = listbox.get(selection[0])

        # Extrahera filnamnet från listraden

        parts = selected_text.split()

        if not parts:

            return

            

        amiga_file = parts[0]  # första ordet = filnamn

        pc_file = filedialog.asksaveasfilename(initialfile=amiga_file)

        if not pc_file:

            return

        try:

            subprocess.run(["xdftool", filename, "read", amiga_file, pc_file], check=True)

            messagebox.showinfo("Klart", f"{amiga_file} exporterades till {pc_file}")

        except Exception as e:

            messagebox.showerror("Fel", f"Kunde inte exportera: {e}")



    tk.Button(btn_frame, text="Lägg till fil", command=add_file).pack(side=tk.LEFT, padx=5)

    tk.Button(btn_frame, text="Exportera fil", command=extract_file).pack(side=tk.LEFT, padx=5)

    tk.Button(btn_frame, text="Uppdatera", command=refresh_contents).pack(side=tk.LEFT, padx=5)

    tk.Button(btn_frame, text="Stäng", command=editor.destroy).pack(side=tk.RIGHT, padx=5)



    refresh_contents()



# -------------------------

# GUI-funktioner

# -------------------------



def select_adf_files():

    """Välj ADF-filer från datorn och lägg till i listan"""

    files = filedialog.askopenfilenames(filetypes=[("Amiga Disk Files", "*.adf")])

    for f in files:

        adf_listbox.insert(tk.END, f)

    update_status()



def remove_selected():

    """Ta bort markerad fil från listan"""

    pos = adf_listbox.curselection()

    if pos:

        adf_listbox.delete(pos[0])

        update_status()



def move_up():

    """Flytta markerad fil uppåt i listan"""

    pos = adf_listbox.curselection()

    if not pos:

        return

    index = pos[0]

    if index > 0:

        text = adf_listbox.get(index)

        adf_listbox.delete(index)

        adf_listbox.insert(index - 1, text)

        adf_listbox.select_set(index - 1)



def move_down():

    """Flytta markerad fil nedåt i listan"""

    pos = adf_listbox.curselection()

    if not pos:

        return

    index = pos[0]

    if index < adf_listbox.size() - 1:

        text = adf_listbox.get(index)

        adf_listbox.delete(index)

        adf_listbox.insert(index + 1, text)

        adf_listbox.select_set(index + 1)



def export_to_usb():

    """Kopiera filer till USB och skapa CFG + lista"""

    if adf_listbox.size() == 0:

        messagebox.showwarning("Tom lista", "Inga ADF-filer att exportera.")

        return

        

    target = filedialog.askdirectory(title="Välj USB-minne")

    if not target:

        return



    # Kontrollera skrivrättigheter

    if not os.access(target, os.W_OK):

        messagebox.showerror("Fel", "Inga skrivrättigheter till USB-minnet.")

        return



    mapping = {}

    success_count = 0

    

    for i in range(adf_listbox.size()):

        src = adf_listbox.get(i)

        if not os.path.isfile(src):

            messagebox.showerror("Fel", f"Filen finns inte längre:\n{src}")

            continue



        dst_name = f"D{i+1:03}.ADF"

        dst = os.path.join(target, dst_name)

        

        try:

            shutil.copy2(src, dst)

            mapping[dst_name] = {

                "original_name": os.path.basename(src),

                "path": src

            }

            success_count += 1

        except Exception as e:

            messagebox.showerror("Fel", f"Kunde inte kopiera {src}: {e}")



    if success_count > 0:

        # Spara mappningen

        try:

            with open(os.path.join(target, "adf_list.json"), "w") as f:

                json.dump(mapping, f, indent=4)

            

            # Skapa enkel FlashFloppy-konfig

            cfg_content = f"""# FlashFloppy konfiguration - Skapad av {APP_NAME}

interface = jc

indexed-mode = yes

display-type = oled-128x32-rotate

# ADF-filer: {success_count}

"""

            with open(os.path.join(target, "FF.CFG"), "w") as cfg:

                cfg.write(cfg_content)

            

            messagebox.showinfo("Klart!", f"{success_count} ADF-filer har kopierats till USB.")

        except Exception as e:

            messagebox.showerror("Fel", f"Kunde inte spara konfiguration: {e}")

    else:

        messagebox.showwarning("Varning", "Inga filer kunde kopieras.")



def import_from_usb():

    """Läs in befintlig lista från USB och visa i GUI"""

    source = filedialog.askdirectory(title="Välj USB-minne")

    if not source:

        return



    list_file = os.path.join(source, "adf_list.json")

    if not os.path.exists(list_file):

        messagebox.showwarning("Ingen lista", "Ingen adf_list.json hittades på USB.")

        return



    try:

        with open(list_file, "r") as f:

            mapping = json.load(f)



        adf_listbox.delete(0, tk.END)



        # Sortera på D001, D002 osv.

        for k in sorted(mapping.keys()):

            item = mapping[k]

            if os.path.exists(item["path"]):

                adf_listbox.insert(tk.END, item["path"])

            else:

                # Försök hitta filen relativt USB-sökvägen

                usb_file = os.path.join(source, k)

                if os.path.exists(usb_file):

                    adf_listbox.insert(tk.END, usb_file)

        

        messagebox.showinfo("Klart", f"Importerade {adf_listbox.size()} filer från USB.")

        update_status()

        

    except Exception as e:

        messagebox.showerror("Fel", f"Kunde inte importera från USB: {e}")



def clear_list():

    """Rensa hela listan"""

    if adf_listbox.size() > 0:

        if messagebox.askyesno("Bekräfta", "Vill du verkligen rensa hela listan?"):

            adf_listbox.delete(0, tk.END)

            update_status()



def show_help():

    """Visa hjälpinformation"""

    help_text = f"""{APP_NAME} {APP_VERSION} - Hjälp



Välkommen till ADF Commander - ditt verktyg för att hantera

Amiga Disk Files för användning med Gotek-enheter.



Funktioner:

• Lägg till ADF: Välj befintliga ADF-filer

• Skapa tom ADF: Skapar oformaterade ADF-filer (901120 bytes)

• Skapa formaterad ADF: Skapar formaterade ADF-filer (kräver xdftool)

• Redigera ADF: Lägg till/ta bort filer i ADF (kräver xdftool)

• Exportera till USB: Kopierar filer med FF.CFG konfiguration

• Importera från USB: Läs in tidigare exporterade listor



Oformaterade ADF-filer:

- Fungerar med alla emulatorer

- Kan formateras i Workbench

- Är pålitliga och enkla att skapa



Formaterade ADF-filer:

- Redo att användas direkt

- Kräver xdftool för att skapas/redigeras



Tips: Använd 'Skapa tom ADF' för nya disketter och formatera

dem sedan i Workbench för bästa resultat!"""

    

    messagebox.showinfo("Hjälp", help_text)



def show_about():

    """Visa information om programmet"""

    about_text = f"""{APP_NAME} {APP_VERSION}



Ett verktyg för att hantera Amiga Disk Files (ADF)

skapat för användning med Gotek-enheter och FlashFloppy.



Funktioner:

- Skapa tomma och formaterade ADF-filer

- Redigera ADF-filers innehåll (med xdftool)

- Hantera ADF-samlingar

- Exportera till USB för Gotek

- Importera från befintliga konfigurationer



Oformaterade ADF-filer är kompatibla med alla

Amiga-emulatorer och kan formateras i Workbench.



Formaterade ADF-filer kräver xdftool för skapande och redigering."""

    

    messagebox.showinfo("Om", about_text)



def update_status(event=None):

    count = adf_listbox.size()

    status_label.config(text=f"Antal ADF-filer: {count}")



# -------------------------

# GUI-uppsättning

# -------------------------



root = tk.Tk()

root.title(f"{APP_NAME} {APP_VERSION}")

root.geometry("800x600")



# Skapa en menyrad

menubar = tk.Menu(root)

root.config(menu=menubar)



# Arkiv-menyn

file_menu = tk.Menu(menubar, tearoff=0)

menubar.add_cascade(label="Arkiv", menu=file_menu)

file_menu.add_command(label="Lägg till ADF-filer", command=select_adf_files)

file_menu.add_command(label="Skapa tom ADF", command=create_blank_adf)

file_menu.add_command(label="Skapa formaterad ADF", command=create_formatted_adf)

file_menu.add_separator()

file_menu.add_command(label="Avsluta", command=root.quit)



# Redigera-menyn

edit_menu = tk.Menu(menubar, tearoff=0)

menubar.add_cascade(label="Redigera", menu=edit_menu)

edit_menu.add_command(label="Redigera ADF-innehåll", command=open_adf_editor)

edit_menu.add_separator()

edit_menu.add_command(label="Ta bort vald", command=remove_selected)

edit_menu.add_command(label="Rensa lista", command=clear_list)



# Verktyg-menyn

tools_menu = tk.Menu(menubar, tearoff=0)

menubar.add_cascade(label="Verktyg", menu=tools_menu)

tools_menu.add_command(label="Exportera till USB", command=export_to_usb)

tools_menu.add_command(label="Importera från USB", command=import_from_usb)

tools_menu.add_separator()

tools_menu.add_command(label="Flytta upp", command=move_up)

tools_menu.add_command(label="Flytta ner", command=move_down)



# Hjälp-menyn

help_menu = tk.Menu(menubar, tearoff=0)

menubar.add_cascade(label="Hjälp", menu=help_menu)

help_menu.add_command(label="Hjälp", command=show_help)

help_menu.add_command(label="Om ADF Commander", command=show_about)



# Main frame

main_frame = tk.Frame(root)

main_frame.pack(fill=tk.BOTH, expand=True, padx=10, pady=10)



# Listbox med scrollbar

list_frame = tk.Frame(main_frame)

list_frame.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)



scrollbar = tk.Scrollbar(list_frame)

scrollbar.pack(side=tk.RIGHT, fill=tk.Y)



adf_listbox = tk.Listbox(list_frame, width=70, height=20, yscrollcommand=scrollbar.set)

adf_listbox.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)

scrollbar.config(command=adf_listbox.yview)



# Knappram

btn_frame = tk.Frame(main_frame)

btn_frame.pack(side=tk.RIGHT, fill=tk.Y, padx=(10, 0))



# Knappar

buttons = [

    ("Lägg till ADF", select_adf_files),

    ("Skapa tom ADF", create_blank_adf),

    ("Skapa formaterad ADF", create_formatted_adf),

    ("Redigera ADF", open_adf_editor),

    ("Ta bort vald", remove_selected),

    ("Flytta upp", move_up),

    ("Flytta ner", move_down),

    ("Rensa lista", clear_list),

    ("Exportera till USB", export_to_usb),

    ("Importera från USB", import_from_usb),

    ("Hjälp", show_help)

]



for text, command in buttons:

    btn = tk.Button(btn_frame, text=text, command=command, width=15)

    btn.pack(fill=tk.X, pady=2)



# Statusrad

status_frame = tk.Frame(root)

status_frame.pack(fill=tk.X, padx=10, pady=5)



status_label = tk.Label(status_frame, text="Antal ADF-filer: 0", relief=tk.SUNKEN, anchor=tk.W)

status_label.pack(fill=tk.X)



adf_listbox.bind("<<ListboxSelect>>", update_status)

root.after(100, update_status)



# Centrera fönstret på skärmen

root.eval('tk::PlaceWindow . center')



root.mainloop()